<?php
/*
Plugin Name: API-97.com WooCommerce Gateway
Description: Accept payments via API-97.com on WooCommerce. Built by <a href="https://api-97.com" target="_blank">API-97.com</a>
Version: 1.0.5
Author: API-97.com | Developer Sharif
Author URI: https://api-97.com
*/

if (!defined('ABSPATH')) exit;

add_action('plugins_loaded', 'api97_wc_gateway_init', 11);

function api97_wc_gateway_init() {
    if ( ! class_exists('WC_Payment_Gateway') ) return;

    class WC_Gateway_API97 extends WC_Payment_Gateway {

        public function __construct() {
            $this->id                 = 'api97';
            $this->icon               = plugins_url('assets/logo.png', __FILE__);
            $this->has_fields         = false;
            $this->method_title       = 'API-97.com Gateway';
            $this->method_description = 'Pay securely via API-97.com (Multi-gateway proxy, UPI, Razorpay, Paytm)';

            $this->init_form_fields();
            $this->init_settings();

            $this->title        = $this->get_option('title');
            $this->description  = $this->get_option('description');
            $this->api_key      = trim($this->get_option('api_key'));
            $this->api_secret   = trim($this->get_option('api_secret'));
            $this->webhook_url  = trim($this->get_option('webhook_url')); // optional; else auto

            add_action('woocommerce_update_options_payment_gateways_' . $this->id, [$this, 'process_admin_options']);

            // Front styles
            add_action('wp_enqueue_scripts', function () {
                wp_register_style('api97-woo-style', plugins_url('assets/style.css', __FILE__), [], '1.0.5');
                wp_enqueue_style('api97-woo-style');
            });

            // Endpoints
            add_action('woocommerce_api_api97_open',    [$this, 'endpoint_open_checkout']); // /?wc-api=api97_open&oid=ID
            add_action('woocommerce_api_api97_confirm', [$this, 'endpoint_confirm']);       // client fallback
            add_action('woocommerce_api_api97_webhook', [$this, 'endpoint_webhook']);       // (optional) if ever used
        }

        public function init_form_fields() {
            $this->form_fields = [
                'enabled' => [
                    'title'   => 'Enable/Disable',
                    'type'    => 'checkbox',
                    'label'   => 'Enable API-97.com Payment Gateway',
                    'default' => 'yes',
                ],
                'title' => [
                    'title'       => 'Gateway Title',
                    'type'        => 'text',
                    'description' => 'Title on Checkout page.',
                    'default'     => 'API-97.com Payment',
                ],
                'description' => [
                    'title'       => 'Description',
                    'type'        => 'textarea',
                    'description' => 'Payment method description.',
                    'default'     => 'Pay securely via API-97.com (UPI, Razorpay, etc.)',
                ],
                'api_key' => [
                    'title'       => 'API-97.com API Key',
                    'type'        => 'text',
                    'description' => '<a href="https://getaways.api-97.com/" target="_blank">Get from Dashboard</a>',
                    'default'     => '',
                ],
                'api_secret' => [
                    'title'       => 'API-97.com API Secret',
                    'type'        => 'password',
                    'description' => 'Get from Dashboard',
                    'default'     => '',
                ],
                'webhook_url' => [
                    'title'       => 'Webhook URL (optional)',
                    'type'        => 'text',
                    'description' => 'Leave blank to auto-use: ' . esc_html( home_url('/?wc-api=api97_webhook') ),
                    'default'     => '',
                ],
            ];
        }

        public function payment_fields() {
            echo '<div style="margin-bottom:7px;font-weight:600;">' . esc_html($this->description) . '</div>';
            echo '<img src="'.esc_url($this->icon).'" alt="API-97.com" style="max-width:120px;border-radius:12px;margin-bottom:8px;">';
        }

        private function relay_url_local()     { return $this->webhook_url ?: home_url('/?wc-api=api97_webhook'); }
        private function open_url($oid)        { return home_url('/?wc-api=api97_open&oid='.(int)$oid); }
        private function confirm_url()         { return home_url('/?wc-api=api97_confirm'); }
        private function gateway_self_webhook() { return 'https://gateways.api-97.com/webhook/callback.php'; }

        /** Main payment step */
        public function process_payment($order_id) {
            $order = wc_get_order($order_id);

            $payload = [
                'api_key'        => $this->api_key,
                'api_secret'     => $this->api_secret,
                'amount'         => number_format((float)$order->get_total(), 2, '.', ''),
                'currency'       => $order->get_currency(),
                'customer_name'  => $order->get_formatted_billing_full_name(),
                'customer_email' => $order->get_billing_email(),
                'redirect_url'   => $order->get_checkout_order_received_url(), // Thank you
                // provider self webhook -> gateway dashboard updates to 'success'
                'relay_url'      => $this->gateway_self_webhook(),
                'user_id'        => (string)$order_id, // metadata
            ];

            $ch = curl_init('https://gateways.api-97.com/api/v1.php');
            curl_setopt($ch, CURLOPT_POST, 1);
            curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($payload));
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
            $response = curl_exec($ch);
            $http     = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);

            $res = json_decode($response, true);
            if (!is_array($res) || ($res['status'] ?? '') !== 'success') {
                wc_add_notice('Payment initiation failed: '. ( $res['msg'] ?? ('HTTP '.$http) ), 'error');
                return ['result'=>'fail'];
            }

            $provider_order_id = $res['rzp']['id'] ?? ($res['order_id'] ?? '');
            $public_key        = $res['public_key'] ?? '';
            $amount_paise      = (int)($res['rzp']['amount'] ?? round($order->get_total()*100));
            $currency          = $res['rzp']['currency'] ?? $order->get_currency();

            if (!$provider_order_id || !$public_key) {
                wc_add_notice('Payment initiation invalid response.', 'error');
                return ['result'=>'fail'];
            }

            update_post_meta($order_id, '_api97_provider_order_id', $provider_order_id);
            update_post_meta($order_id, '_api97_public_key',        $public_key);
            update_post_meta($order_id, '_api97_amount_paise',      $amount_paise);
            update_post_meta($order_id, '_api97_currency',          $currency);

            $order->update_status('pending', 'API-97 order created: '.$provider_order_id);

            return [
                'result'   => 'success',
                'redirect' => $this->open_url($order_id),
            ];
        }

        /** Minimal, no extra HTML — auto open Razorpay and then go to Thank You AFTER confirm completes */
        public function endpoint_open_checkout() {
            $order_id = absint($_GET['oid'] ?? 0);
            if (!$order_id) wp_die('Missing order id');

            $order        = wc_get_order($order_id);
            $public_key   = get_post_meta($order_id, '_api97_public_key', true);
            $rp_order_id  = get_post_meta($order_id, '_api97_provider_order_id', true);
            $rp_amount    = (int) get_post_meta($order_id, '_api97_amount_paise', true);
            $currency     = get_post_meta($order_id, '_api97_currency', true) ?: 'INR';

            if (!$order || !$public_key || !$rp_order_id) wp_die('Order not ready');

            $thankyou = $order->get_checkout_order_received_url();
            $confirm  = $this->confirm_url();
            ?>
<!doctype html>
<html>
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <meta http-equiv="Cache-Control" content="no-store, no-cache, must-revalidate, max-age=0"/>
  <meta http-equiv="Pragma" content="no-cache"/>
  <meta http-equiv="Expires" content="0"/>
  <script src="https://checkout.razorpay.com/v1/checkout.js" async></script>
  <style>html,body{margin:0;background:#fff}</style>
</head>
<body>
<script>
(function(){
  var thankyou = <?php echo json_encode($thankyou); ?>;
  var confirmU = <?php echo json_encode($confirm); ?>;
  var orderId  = <?php echo json_encode($order_id); ?>;

  function sendConfirm(status){
    if (!status) return Promise.resolve();
    try{
      var controller = ('AbortController' in window) ? new AbortController() : null;
      var signal = controller ? controller.signal : undefined;
      var to = setTimeout(function(){ if (controller) { try{controller.abort()}catch(e){} } }, 3500);
      return fetch(confirmU, {
        method:'POST',
        headers:{'Content-Type':'application/x-www-form-urlencoded'},
        body:new URLSearchParams({order_id: orderId, status: status}),
        keepalive:true,
        signal: signal
      }).catch(function(){}).finally(function(){ clearTimeout(to); });
    }catch(e){ return Promise.resolve(); }
  }

  function goThankYou(status){ sendConfirm(status).finally(function(){ window.location.replace(thankyou); }); }

  function openRzp(){
    try{
      var r = new Razorpay({
        key:       <?php echo json_encode($public_key); ?>,
        order_id:  <?php echo json_encode($rp_order_id); ?>,
        amount:    <?php echo json_encode($rp_amount); ?>,
        currency:  <?php echo json_encode($currency); ?>,
        name: "API-97 Payment",
        description: "Woo Order <?php echo esc_js($order->get_order_number()); ?>",
        handler: function(){ goThankYou('success'); },
        modal: { ondismiss: function(){ goThankYou('failed'); } }
      });
      r.on && r.on('payment.failed', function(){ goThankYou('failed'); });
      r.open();
    } catch(e){ goThankYou(''); }
  }

  if (window.Razorpay) openRzp();
  document.addEventListener('DOMContentLoaded', function(){ if (window.Razorpay) openRzp(); });
  window.addEventListener('load', function(){ if (window.Razorpay) openRzp(); });
  (function waitAndOpen(tries){
    if (window.Razorpay) return;
    if (tries<=0) return goThankYou('');
    setTimeout(function(){ if (window.Razorpay) openRzp(); else waitAndOpen(tries-1); }, 300);
  })(20);
})();
</script>
<noscript><meta http-equiv="refresh" content="0;url=<?php echo esc_url($thankyou); ?>"></noscript>
</body>
</html>
            <?php
            exit;
        }

        /** Client fallback confirm (idempotent) */
        public function endpoint_confirm() {
            $order_id = absint($_POST['order_id'] ?? 0);
            $status   = strtolower($_POST['status'] ?? '');
            if (!$order_id) { status_header(400); echo 'missing'; exit; }

            $order = wc_get_order($order_id);
            if (!$order) { status_header(404); echo 'no order'; exit; }
            if ($order->is_paid()) { echo 'ok'; exit; }

            if ($status === 'success' || $status === 'paid' || $status === 'captured') {
                // Mark paid: sets status to processing/completed based on items
                $order->payment_complete();

                // Ensure downloads are granted immediately (idempotent)
                if (function_exists('wc_downloadable_product_permissions')) {
                    wc_downloadable_product_permissions($order_id, true);
                }

                $order->add_order_note('API-97 client-confirmed success.');
                WC()->cart && WC()->cart->empty_cart();
            } elseif ($status === 'failed' || $status === 'failure') {
                $order->update_status('failed', 'API-97 client-confirmed failure.');
            } else {
                $order->update_status('on-hold', 'API-97 client-confirmed pending/created.');
            }
            echo 'ok';
            exit;
        }

        /** (Optional) Webhook if ever pointed to us; keeps compatibility */
        public function endpoint_webhook() {
            $raw  = file_get_contents('php://input');
            $data = json_decode($raw, true);
            if (!$data) $data = $_POST;

            $prov_id = $data['order_id'] ?? ($data['rzp_order_id'] ?? '');
            $status  = strtolower($data['status'] ?? '');

            if (!$prov_id) { status_header(400); echo 'missing order_id'; exit; }

            $orders = wc_get_orders([
                'limit'      => 1,
                'meta_key'   => '_api97_provider_order_id',
                'meta_value' => $prov_id,
                'return'     => 'objects',
            ]);
            if (empty($orders)) { status_header(404); echo 'order not found'; exit; }

            $order = $orders[0];
            if ($order->is_paid()) { echo 'success'; exit; }

            if ($status === 'paid' || $status === 'captured' || $status === 'success') {
                $order->payment_complete();

                // Ensure downloads are granted (idempotent)
                if (function_exists('wc_downloadable_product_permissions')) {
                    wc_downloadable_product_permissions($order->get_id(), true);
                }

                $order->add_order_note('API-97 webhook: success ('.$prov_id.')');
                WC()->cart && WC()->cart->empty_cart();
            } elseif ($status === 'failed' || $status === 'failure') {
                $order->update_status('failed', 'API-97 webhook: failed ('.$prov_id.')');
            } else {
                $order->update_status('on-hold', 'API-97 webhook: '.$status.' ('.$prov_id.')');
            }

            echo 'success';
            exit;
        }
    }

    add_filter('woocommerce_payment_gateways', function($methods){
        $methods[] = 'WC_Gateway_API97'; return $methods;
    });
}

add_filter('plugin_action_links_' . plugin_basename(__FILE__), function($links) {
    $links[] = '<a style="font-weight:700;color:#19b23b" href="https://gateways.api-97.com/" target="_blank">API-97.com</a>';
    $links[] = '<span style="color:#3f5efb;font-weight:600;">Developer: Sharif</span>';
    return $links;
});
